document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const batchExtractForm = document.getElementById('batchExtractForm');
    const youtubeUrlsInput = document.getElementById('youtubeUrls');
    const formatSelect = document.getElementById('formatSelect');
    const languageSelect = document.getElementById('languageSelect');

    const extractBtn = document.getElementById('extractBtn');
    const loadingIndicator = document.getElementById('loadingIndicator');
    const resultSection = document.getElementById('resultSection');
    const errorSection = document.getElementById('errorSection');
    const batchResults = document.getElementById('batchResults');
    const errorMessage = document.getElementById('errorMessage');
    const successCount = document.getElementById('successCount');
    const errorCount = document.getElementById('errorCount');

    // Form submission handler
    batchExtractForm.addEventListener('submit', function(e) {
        e.preventDefault();

        const youtubeUrlsText = youtubeUrlsInput.value.trim();
        if (!youtubeUrlsText) {
            showError('Please enter at least one YouTube URL');
            return;
        }

        // Parse URLs (one per line)
        const urls = youtubeUrlsText.split('\n')
            .map(url => url.trim())
            .filter(url => url.length > 0);

        if (urls.length === 0) {
            showError('Please enter at least one valid YouTube URL');
            return;
        }

        const format = formatSelect.value;
        const language = languageSelect.value;


        extractBatchSubtitles(urls, format, language);
    });

    // Extract batch subtitles function
    async function extractBatchSubtitles(urls, format, language) {
        // Show loading indicator
        hideAllSections();
        loadingIndicator.classList.remove('hidden');

        try {
            const response = await fetch('/api/batch-extract', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    urls,
                    format,
                    language
                })
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || 'Failed to extract subtitles');
            }

            // 처리 중인 작업이 있는지 확인
            if (data.processing && data.processing.length > 0) {
                console.log(`${data.processing.length} videos are still being processed...`);
                // 지정된 시간 후에 다시 시도
                const retryAfter = data.retry_after || 5;

                // 일부 결과가 있는 경우 표시
                if (data.results && data.results.length > 0) {
                    loadingIndicator.innerHTML = `
                        <p>${data.message || 'Some videos are still being processed...'}</p>
                        <p>Showing ${data.results.length} completed results. Retrying for remaining ${data.processing.length} videos in ${retryAfter} seconds...</p>
                        <div class="progress-container">
                            <div class="progress-bar" style="width: ${Math.round(data.results.length / (data.results.length + data.processing.length) * 100)}%"></div>
                        </div>
                    `;

                    // 결과 표시
                    displayBatchResults(data);
                } else {
                    loadingIndicator.innerHTML = `
                        <p>${data.message || 'Processing your request...'}</p>
                        <p>Retrying in ${retryAfter} seconds...</p>
                        <div class="progress-container">
                            <div class="progress-bar" style="width: 0%"></div>
                        </div>
                    `;
                }

                // 지정된 시간 후에 다시 시도
                setTimeout(() => {
                    extractBatchSubtitles(urls, format, language);
                }, retryAfter * 1000);
                return;
            }

            // Display the results
            displayBatchResults(data);

            // Show result section
            hideAllSections();
            resultSection.classList.remove('hidden');

        } catch (error) {
            showError(error.message);
        }
    }

    // Display batch results
    function displayBatchResults(data) {
        // Update counts
        successCount.textContent = data.total_processed;
        errorCount.textContent = data.total_errors;

        // Clear previous results
        batchResults.innerHTML = '';

        // Add successful results
        if (data.results.length > 0) {
            const successSection = document.createElement('div');
            successSection.className = 'success-section';

            const successHeader = document.createElement('h3');
            successHeader.innerHTML = '<i class="fas fa-check-circle"></i> Successful Extractions';
            successSection.appendChild(successHeader);

            data.results.forEach(result => {
                const resultItem = document.createElement('div');
                resultItem.className = 'result-item';

                const resultHeader = document.createElement('div');
                resultHeader.className = 'result-item-header';

                const title = document.createElement('h4');
                title.textContent = result.video_title;
                resultHeader.appendChild(title);



                const actions = document.createElement('div');
                actions.className = 'result-item-actions';

                const downloadBtn = document.createElement('button');
                downloadBtn.className = 'btn-download';
                downloadBtn.innerHTML = '<i class="fas fa-download"></i> Download';
                downloadBtn.addEventListener('click', () => downloadSubtitles(result));
                actions.appendChild(downloadBtn);

                const copyBtn = document.createElement('button');
                copyBtn.className = 'btn-copy';
                copyBtn.innerHTML = '<i class="fas fa-copy"></i> Copy';
                copyBtn.addEventListener('click', () => copyToClipboard(result.transcript, copyBtn));
                actions.appendChild(copyBtn);

                resultItem.appendChild(resultHeader);
                resultItem.appendChild(actions);

                const preview = document.createElement('div');
                preview.className = 'transcript-preview';

                // Create a preview of the transcript (first 100 characters)
                const previewText = result.transcript.length > 100
                    ? result.transcript.substring(0, 100) + '...'
                    : result.transcript;

                preview.textContent = previewText;
                resultItem.appendChild(preview);

                successSection.appendChild(resultItem);
            });

            batchResults.appendChild(successSection);
        }

        // Add errors
        if (data.errors.length > 0) {
            const errorSection = document.createElement('div');
            errorSection.className = 'error-results-section';

            const errorHeader = document.createElement('h3');
            errorHeader.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Errors';
            errorSection.appendChild(errorHeader);

            data.errors.forEach(error => {
                const errorItem = document.createElement('div');
                errorItem.className = 'error-item';

                const url = document.createElement('div');
                url.className = 'error-url';
                url.textContent = error.url;
                errorItem.appendChild(url);

                const message = document.createElement('div');
                message.className = 'error-message';
                message.textContent = error.error;
                errorItem.appendChild(message);

                errorSection.appendChild(errorItem);
            });

            batchResults.appendChild(errorSection);
        }
    }

    // Download subtitles function
    async function downloadSubtitles(result) {
        try {
            const response = await fetch('/api/download', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    transcript: result.transcript,
                    filename: result.filename
                })
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'Failed to download subtitles');
            }

            // Create a blob from the response
            const blob = await response.blob();

            // Create a download link
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.style.display = 'none';
            a.href = url;
            a.download = result.filename;

            // Append to the document and trigger the download
            document.body.appendChild(a);
            a.click();

            // Clean up
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);

        } catch (error) {
            showError(error.message);
        }
    }

    // Copy to clipboard function
    function copyToClipboard(text, button) {
        navigator.clipboard.writeText(text)
            .then(() => {
                // Show temporary success message
                const originalText = button.innerHTML;
                button.innerHTML = '<i class="fas fa-check"></i> Copied!';

                setTimeout(() => {
                    button.innerHTML = originalText;
                }, 2000);
            })
            .catch(err => {
                showError('Failed to copy: ' + err.message);
            });
    }

    // Helper function to show error
    function showError(message) {
        hideAllSections();
        errorMessage.textContent = message;
        errorSection.classList.remove('hidden');
    }

    // Helper function to hide all sections
    function hideAllSections() {
        loadingIndicator.classList.add('hidden');
        resultSection.classList.add('hidden');
        errorSection.classList.add('hidden');
    }
});
