document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const extractForm = document.getElementById('extractForm');
    const youtubeUrlInput = document.getElementById('youtubeUrl');
    const formatSelect = document.getElementById('formatSelect');
    const languageSelect = document.getElementById('languageSelect');

    const extractBtn = document.getElementById('extractBtn');
    const loadingIndicator = document.getElementById('loadingIndicator');
    const resultSection = document.getElementById('resultSection');
    const errorSection = document.getElementById('errorSection');
    const videoTitle = document.getElementById('videoTitle');
    const transcriptContent = document.getElementById('transcriptContent');
    const errorMessage = document.getElementById('errorMessage');
    const downloadBtn = document.getElementById('downloadBtn');
    const copyBtn = document.getElementById('copyBtn');

    // Current transcript data
    let currentTranscriptData = null;

    // Form submission handler
    extractForm.addEventListener('submit', function(e) {
        e.preventDefault();

        const youtubeUrl = youtubeUrlInput.value.trim();
        const format = formatSelect.value;
        const language = languageSelect.value;


        if (!youtubeUrl) {
            showError('Please enter a YouTube URL');
            return;
        }

        extractSubtitles(youtubeUrl, format, language);
    });

    // Extract subtitles function
    async function extractSubtitles(url, format, language) {
        // Show loading indicator
        hideAllSections();
        loadingIndicator.classList.remove('hidden');
        loadingIndicator.innerHTML = '<div class="spinner"></div><p>Extracting subtitles...</p>';

        try {
            const response = await fetch('/api/extract', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ url, format, language })
            });

            const data = await response.json();

            if (!response.ok) {
                // Handle specific error cases
                if (data.error === 'YouTube cookies not configured') {
                    // Special handling for cookie configuration error
                    const errorDetails = `
                        <div class="error-details">
                            <p><strong>${data.message || 'YouTube requires authentication.'}</strong></p>
                            <p>${data.details || ''}</p>
                            <p class="solution"><strong>Solution:</strong> ${data.solution || 'Set up YouTube cookies to bypass anti-bot measures.'}</p>
                            <p>See the <a href="YOUTUBE_COOKIES_GUIDE.md" target="_blank">YouTube Cookies Guide</a> for instructions.</p>
                        </div>
                    `;
                    showError('YouTube Cookies Not Configured', errorDetails);
                    return;
                } else {
                    throw new Error(data.error || 'Failed to extract subtitles');
                }
            }

            // 처리 중인 상태 확인
            if (data.status === 'processing') {
                console.log('Request is being processed in the background...');
                // 지정된 시간 후에 다시 시도
                const retryAfter = data.retry_after || 5;
                let message = `<p>${data.message || 'Processing your request...'}</p><p>Retrying in <span id="countdown">${retryAfter}</span> seconds...</p>`;

                // Add note about cookies if provided
                if (data.note) {
                    message += `<p class="note"><strong>Note:</strong> ${data.note}</p>`;
                }

                loadingIndicator.innerHTML = message;

                // Countdown timer
                let countdown = retryAfter;
                const countdownElement = document.getElementById('countdown');
                const countdownInterval = setInterval(() => {
                    countdown--;
                    if (countdownElement) {
                        countdownElement.textContent = countdown;
                    }
                    if (countdown <= 0) {
                        clearInterval(countdownInterval);
                    }
                }, 1000);

                // 지정된 시간 후에 다시 시도
                setTimeout(() => {
                    clearInterval(countdownInterval);
                    extractSubtitles(url, format, language);
                }, retryAfter * 1000);
                return;
            }

            // 캐싱된 결과인지 표시
            if (data.cached) {
                console.log('Retrieved from cache');
                const timestamp = data.timestamp || 'unknown time';
                videoTitle.innerHTML = `${data.video_title} <span class="cached-info">(Cached: ${timestamp})</span>`;
            } else {
                videoTitle.textContent = data.video_title;
            }

            // Store the current transcript data
            currentTranscriptData = data;

            // Display the results
            transcriptContent.textContent = data.transcript;

            // Show result section
            hideAllSections();
            resultSection.classList.remove('hidden');

        } catch (error) {
            showError('Error', error.message);
        }
    }

    // Download button handler
    downloadBtn.addEventListener('click', async function() {
        if (!currentTranscriptData) return;

        try {
            const response = await fetch('/api/download', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    transcript: currentTranscriptData.transcript,
                    filename: currentTranscriptData.filename
                })
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'Failed to download subtitles');
            }

            // Create a blob from the response
            const blob = await response.blob();

            // Create a download link
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.style.display = 'none';
            a.href = url;
            a.download = currentTranscriptData.filename;

            // Append to the document and trigger the download
            document.body.appendChild(a);
            a.click();

            // Clean up
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);

        } catch (error) {
            showError(error.message);
        }
    });

    // Copy button handler
    copyBtn.addEventListener('click', function() {
        if (!currentTranscriptData) return;

        // Copy to clipboard
        navigator.clipboard.writeText(currentTranscriptData.transcript)
            .then(() => {
                // Show temporary success message
                const originalText = copyBtn.innerHTML;
                copyBtn.innerHTML = '<i class="fas fa-check"></i> Copied!';

                setTimeout(() => {
                    copyBtn.innerHTML = originalText;
                }, 2000);
            })
            .catch(err => {
                showError('Failed to copy: ' + err.message);
            });
    });

    // Helper function to show error
    function showError(title, message) {
        hideAllSections();

        // If only one parameter is provided, use it as the message
        if (!message) {
            message = title;
            title = 'Error';
        }

        // Check if message is HTML content
        if (message.includes('<')) {
            errorMessage.innerHTML = `<h3>${title}</h3>${message}`;
        } else {
            errorMessage.innerHTML = `<h3>${title}</h3><p>${message}</p>`;
        }

        errorSection.classList.remove('hidden');
    }

    // Helper function to hide all sections
    function hideAllSections() {
        loadingIndicator.classList.add('hidden');
        resultSection.classList.add('hidden');
        errorSection.classList.add('hidden');
    }

    // Initialize with a sample URL if provided in the query string
    const urlParams = new URLSearchParams(window.location.search);
    const sampleUrl = urlParams.get('url');
    if (sampleUrl) {
        youtubeUrlInput.value = sampleUrl;
    }
});
